#include <stdio.h>
#include <stdlib.h>
#include <sys/resource.h>
#ifndef GCC
#include <c_asm.h>
#endif
#include "memmap.h"
#include "phibutil.h"
#include "g6util.h"

#define NCLUSTERS (4)
#define DEVID (3)

#define MAXSIZE (0x8000)    /*max data size (=0x20000(=128k) by BYTE)*/
#define SETCOUNT (0x8000)   /*set counter value  (=0x20000 by BYTE)*/
#define GETCOUNT (0x10000)  /*get counter value (=0x40000 by BYTE)*/
#define LINK1BASE (0x10000)  /*output to link1 (=0x40000 by BYTE)*/
#define LINK2BASE (0x18000)  /*output to link2 (=0x60000 by BYTE)*/	     
#define DPRAMBASE (0x20000)  /*read from DPram (=0x80000 by BYTE)*/
#define WRITEBASE (0X00)    /*write address offset by word*/

unsigned int *hib[NCLUSTERS];


int linkopen(int boardid)
{
    hib[boardid] = (unsigned int*)TBopen(boardid);
    if (((int)hib[boardid]) == 0){
	return -1;
    }
    fprintf(stderr,"linkopen returns 0 \n");
    return 0;
}

void linkclose(int boardid)
{
    TBterm(boardid);
}

void reset_board(int boardid)
{
       unsigned int tempadr;
       unsigned int tempdata;
       unsigned int writedata;
       unsigned int mask;
       unsigned int i;
       
       tempadr = 0x6c;

       fprintf(stderr,"g6utils -- resetting HIB %d\n", boardid);
       /*       linkopen(boardid);*/
       mask = 0x00ffffff;
       tempdata = (TBregRead(boardid,tempadr) & mask);
       writedata = (0xd8000000|tempdata);
       TBregWrite(boardid,tempadr,writedata);

       while((TBregRead(boardid,tempadr)&0xff000000)!=0xd8000000){}
       for(i=0;i<100000;i++){
       }
       tempdata = (TBregRead(boardid,tempadr) & mask);
       writedata = (0xb8000000|tempdata);
       TBregWrite(boardid,tempadr,writedata);

       while((TBregRead(boardid,tempadr)&0xff000000)!=0xb8000000){}
       for(i=0;i<100000;i++){
       }
       tempdata = (TBregRead(boardid,tempadr) & mask);
       writedata = (0x98000000|tempdata);
       TBregWrite(boardid,tempadr,writedata);

       /*       linkclose(boardid);*/
       
}


int linkwrite(int boardid,int link,int size, unsigned int * buf)
     /* boardid: index of G6HIB (if there are multiple boards),
	starts from 0*/
     /* link: this is BAD name... 0: internal registers
	                         1,2: links 1, 2 */
{
    int i;
    int adrbase;
    int errcode =0;

    adrbase = setbaseadr(link);
    for(i=0;i<size;i++){
#if INTERNAL_OUT0
	    if (size > 3)fprintf(stderr,"Sending data %x to Link %x\n", buf[i], link);
#endif
	hib[boardid][adrbase+WRITEBASE+i] = buf[i];
    }
    return(errcode);
}

void linkwrite_oneword(int boardid,int link,unsigned int data)
{
    linkwrite(boardid,link,1, &data);
    MB;
}

void linkwrite_onedata(int boardid,
		       int link,
		       unsigned int address,
		       unsigned int data)
{
    unsigned int buf[3];
    buf[0] = address;
    buf[1] = 1;
    buf[2] = data;
#ifdef INTERNAL_OUTX
    fprintf(stderr,"linkwrite_onedata %x %x\n", address, data);
#endif    
    linkwrite(boardid,link,3, buf);
    MB;
}

void dpramdump(int boardid,int size)
{
    int i;
    int adrbase;
    unsigned int errcode = 0;
    
    adrbase = DPRAMBASE;
    for(i=0;i<size;i++){
	printf("dpram[%d]=%x\n",i, ((int *)hib[boardid])[adrbase+i]);
    }
}
int linkread1(int boardid,int size,unsigned int *buf)
{
    register unsigned long v0,v1,v2,v3;
    int i;
    int adrbase,lsize;
    unsigned int errcode = 0;
    
    adrbase = DPRAMBASE>>1;
    lsize=size>>1;
    
    for(i=0;i<lsize;i++){
	v0 = ((long *)hib[boardid])[adrbase+i];
	((long *)buf)[i] = v0;
    }
    return(errcode);
}
int linkread2(int boardid,int size,unsigned int *buf)
{
    register unsigned long v0,v1,v2,v3;
    int i;
    int adrbase,lsize;
    unsigned int errcode = 0;
    
    adrbase = DPRAMBASE>>1;
    lsize=size>>1;
    
    for(i=0;i<lsize;i+=2){
	v0 = ((long *)hib[boardid])[adrbase+i+0];
	v1 = ((long *)hib[boardid])[adrbase+i+1];
	((long *)buf)[i+0] = v0;
	((long *)buf)[i+1] = v1;
    }
    return(errcode);
}

int linkread_with_offset(int boardid,
			 int offset,
			 int size,
			 unsigned int *buf)
{
    register unsigned long v0,v1,v2,v3;
    int i;
    int adrbase,lsize;
    register unsigned long * memp;
    register unsigned long * memp0;
    register unsigned long * bufp;
    int memoffset = offset>>1;
    int memoffset_mask = (MAXSIZE >>1)-1;
    unsigned int errcode = 0;
    adrbase = (DPRAMBASE)>>1;
    memp0 = ((unsigned long *)hib[boardid])+ adrbase;
    lsize=size>>1;
    bufp = (unsigned long *) buf;
    for(i=0;i<lsize;i+=4,  bufp += 4){
	memp = memp0 + memoffset;
	v0 = *memp;
	v1 = *(memp+1);
	v2 = *(memp+2);
	v3 = *(memp+3);
	memoffset += 4;
	memoffset &= memoffset_mask;
	*bufp = v0;
	*(bufp+1) = v1;
	*(bufp+2) = v2;
	*(bufp+3)= v3;
    }
    return(errcode);
}

int linkread(int boardid,
	     int size,
	     unsigned int *buf)
{
    return linkread_with_offset(boardid,0, size, buf);
}



int counterclear(int boardid)
{
    int errcode=0;

    TBmemWrite(boardid,SETCOUNT,0x0);
    if (TBmemRead(boardid,GETCOUNT) != 0){
	errcode = 0x16;
    }
    return(errcode);
}

int g6hib_getcounter(int boardid)
{
    unsigned int m;
    m=TBmemRead(boardid,0x10000)>>2;
    return (int) m;
}

void g6hib_printcounter(int board)
{
    fprintf(stderr,"G6HIB counter = %x\n", g6hib_getcounter(board));
}



int varcheckw(int boardid,int link,int size)
{
    int errcode = 0;
    
    errcode = idcheck(boardid,errcode);
    errcode = linkcheck(link,errcode);
    errcode = sizecheck(size,errcode);
    errcode = countercheck(boardid,errcode);
    return (errcode);
}

int varcheckr(int boardid,int size)
{
    int errcode = 0;
    
    errcode = idcheck(boardid,errcode);
    errcode = sizecheck(size,errcode);
    errcode = countercheck(boardid,errcode);
    return (errcode);
}

int idcheck(int boardid,int errcode)
{
    if ((boardid < 0) || (boardid > DEVID)){
	errcode = errcode | 0x1;
    }
    return (errcode);
}

int linkcheck(int link,int errcode)
{
    if ((link <1) || (link > 2)){
	errcode = errcode | 0x2;
    }
    return (errcode);
}

int sizecheck(int size,int errcode)
{
    if (size > MAXSIZE){
	errcode = errcode | 0x4;
    }
    return (errcode);
}

int countercheck(int boardid,int errcode)
{
    if (counterclear(boardid) != 0){
	errcode = errcode | 0x8;
    }
    return (errcode);
}

int setbaseadr(int link)
{
    int adrbase;
    
    if (link == 1){
	adrbase = LINK1BASE;
    } else if (link == 2) {
        adrbase = LINK2BASE;
    } else {
	adrbase = 0;
    }
    return (adrbase);
}


void g6reset(unsigned int boardid)
{

    linkwrite_oneword(boardid,0,0xFFFFFFFF);
    linkwrite_oneword(boardid,0,0xFFFFF0FF);
    linkwrite_oneword(boardid,0,0xFFFFFFFF);
}

void g6setledmode(unsigned int boardid,
		  unsigned int mode)
     /* mode = 1: programmable
	mode = 0: speedometer (I believe*/
{
    unsigned int link = IPLINK;
    linkwrite_oneword(boardid,link,PBLEDMODEADR);
    linkwrite_oneword(boardid,link,0x1);
    linkwrite_oneword(boardid,link,mode);
}

void g6_led_test(unsigned int boardid)
{
    unsigned int link = IPLINK;
    unsigned int data;
    for (data = 1; data < 0x2000; data = data << 1){
	linkwrite_oneword(boardid,link,PBLEDDATAADR);
	linkwrite_oneword(boardid,link,0x1);
	linkwrite_oneword(boardid,link,data);
	usleep(10000);
    }
}
void g6_led_set(unsigned int boardid, unsigned int data)
{
    unsigned int link = IPLINK;
    linkwrite_oneword(boardid,link,PBLEDDATAADR);
    linkwrite_oneword(boardid,link,0x1);
    linkwrite_oneword(boardid,link,data);
}


void set_nconfig(int board,
		 int link,
		 unsigned int address,
		 unsigned int nconfigbits,
		 unsigned int state)
     
{
    unsigned int data;
    data = 0xfffffffe;
    if (state == 0){
	/* state == 0 means one should set specifined bits to LOW */
	data &= ~(nconfigbits <<2);
    }
#ifdef INTERNAL_OUT
    fprintf(stderr,"set_nconfig %x %x %x\n", nconfigbits, state, data);
#endif    
    /* note that if state = 1, there is no need to modify data from all 1 */
    linkwrite_onedata(board, link, address, data);
}
void send_bit(int board,
		 int link,
		 unsigned int address,
		 unsigned int bit)
     
{
    unsigned int data;
    data = 0xffffffff;
    data &= (0xfffffffc | ((bit & 1) <<1));
    linkwrite_onedata(board, link, address, data);
    data |= 1;
    linkwrite_onedata(board, link, address, data);
}
   


/* send_fpga_data
   address: 0x420 for fofpga
   data: 
      LSB:  dclk
        1:  data0
        2-6: nconfig

   file format: ttf (comma-separatted decimal byte data)
   bit order: should send lsb first
 */
int send_fpga_data(int board,
		   int link,
		   int address,
		   int nconfigbits,
		   char * file_name)
{
    FILE * fin;
    int data, code;
    int datacount;
    /*see if the data file name is set */
    if(file_name == NULL) return -1;
#if INTERNAL_OUT
    printf("(send_fpga_daga) file to open: %s\n", file_name);
#endif
    /* see if data file exists */
    fin = fopen(file_name,"r");
    if(fin == NULL){
	fprintf(stderr,"(send_fpga_data) failed to open defect file %s\n",
		file_name);
	return -1;
    }
    /* data file exists*/

    set_nconfig(board, link, address, nconfigbits,0);
    usleep(1);
    set_nconfig(board, link, address, nconfigbits,1);
    usleep(1);
    datacount = 0;
    while(fscanf(fin,"%d,",&data)==1){
	int i;
	for(i=0;i<8;i++){
	    send_bit(board,link,address,data);
	    data = data >>1;
	}
	datacount++;
	
	if ((datacount & 0xfff) == 0){
	    fprintf(stderr,"datacount = %x\n", datacount);
	    g6_led_set(board, datacount>>8);
	}
    }
    fclose(fin);
    return 0;
}



int g6open(unsigned int boardid)
{
    unsigned int link,data, status;
    int iret;
    link = IPLINK;
#ifdef INTERNAL_OUT    
    fprintf(stderr,"(g6open) Output from board %d Link %d\n", boardid,link);
#endif    
    iret = linkopen(boardid);
    if (iret != 0){
#ifdef INTERNAL_OUT    
	fprintf(stderr,"(g6open) failed...\n");
#endif    
	return  -1;
    }

    g6reset(boardid);

    g6setledmode(boardid, PBLED_PROGRAM_MODE);
    g6_led_test(boardid);
    /* the following send same configuration file to all of the five FPGAs     
    send_fpga_data(boardid, link, 0x420, 31, "fo_unit_with_fifo.ttf");
    */
    
    send_fpga_data(boardid, link, 0x420, 1,
		   "/usr2/makino/src/grape6board/processor_board/fo_unit_thru2-3.ttf");
    send_fpga_data(boardid, link, 0x420, 28,
		   "/usr2/makino/src/grape6board/processor_board/fo_unit_thru2-2.ttf");
    send_fpga_data(boardid, link, 0x420, 2,
		   "/usr2/makino/src/grape6board/processor_board/fo_unit_dum.ttf");
    
    g6_led_test(boardid);
}

#define MAXBURST 8192    


void read_and_compare_result(int board,
			     unsigned int  fodatabuf[],
			     unsigned int fodatacount)
{
    unsigned int g6data[MAXBURST];
    unsigned int ng6data;
    int i;
    
    ng6data = g6hib_getcounter(board);
    fprintf(stderr,"(read and compare results, ndata = host:%d g6:%d\n", fodatacount, ng6data);
    if (ng6data > MAXBURST) ng6data = MAXBURST;
    linkread(board,ng6data,g6data);
    for(i = 0; i<fodatacount;i++){
	fprintf(stderr,"%4d H:%8x G:%8x %8x\n", i, fodatabuf[i],g6data[i],
		fodatabuf[i] ^g6data[i]);
    }
}

/*
  g6_send_testpattern
  file format: chiptest pattern file
 */
int g6_send_testpattern(int board,
			char * file_name)
{
    FILE * fin;
    char linebuf[1024];
    unsigned int databuf[8192];
    unsigned int datacount;
    unsigned int linkprev;
    int data, code;
    unsigned int fodatabuf[8192];
    unsigned int fodatacount = 0;
    unsigned int fostate = 0;

    /*see if the data file name is set */
    if(file_name == NULL) return -1;
#if INTERNAL_OUT
    printf("(g6_send_testpattern) file to open: %s\n", file_name);
#endif
    /* see if data file exists */
    fin = fopen(file_name,"r");
    if(fin == NULL){
	fprintf(stderr,"(g6_send_testpattern) failed to open defect file %s\n",
		file_name);
	return -1;
    }
    /* data file exists*/
    
    datacount = 0;
    linkprev = -1;
    counterclear(board);
    g6hib_printcounter(board);
    while(fgets(linebuf, 1023,fin)!=NULL){
	unsigned long ldata;
	unsigned int  data;
	int link;
	char header[128], header2[128];
	sscanf(linebuf, "%s %lx", header, &ldata);
	data = (unsigned int) (ldata & 0xffffffff);
	link = -1;
	if(strcmp("IPHST", header)==0){
	    link = IPLINK;
	}else	if(strcmp("JPHST", header)==0){
	    link = JPLINK;
	}

	if ((link == -1) && (datacount > 0)){
	    linkwrite(board,linkprev, datacount,databuf);
	    datacount = 0;
	}
	if ((link == IPLINK) ||(link == JPLINK)){
	    if (link != linkprev){
		if ((linkprev != -1) && (datacount > 0)){
		    linkwrite(board,linkprev, datacount,databuf);
		}
		linkprev = link;
		datacount = 0;
	    }
	    if ((link == JPLINK) && (data == 0xffc00)) data = 0;
#if INTERNAL_OUT
	    fprintf(stderr,"Sending data %x to Link %x --- %s", data, link, linebuf);
#endif
	    databuf[datacount] = data;
	    datacount ++;
	    /*	    linkwrite_oneword(board, link, data);*/
	}
	    
	if(strcmp("FOHST", header)==0){
#if INTERNAL_OUT0
	    fprintf(stderr,"Expected  data: %s", linebuf);
#endif
	    fostate = 1;
	    if((linebuf[16] == '0')&&(linebuf[17] == '0')){
		fodatabuf[fodatacount] = data;
		fodatacount ++;
#if INTERNAL_OUT
		fprintf(stderr,"Valid expected  data: %s", linebuf);
#endif
	    }else{
#if INTERNAL_OUT
		fprintf(stderr,"Invalid expected  data: %s", linebuf);
#endif
	    }
	}else{
	    if(fostate == 1){
		read_and_compare_result(board, fodatabuf, fodatacount);
		counterclear(board);
		fodatacount = 0;
		fostate = 0;
	    }
	}

	
    }
    if ((linkprev != -1) && (datacount > 0)){
	linkwrite(board,linkprev, datacount,databuf);
    }
    if(fostate == 1){
	read_and_compare_result(board, fodatabuf, fodatacount);
	counterclear(board);
	fodatacount = 0;
    }
    fclose(fin);
    return 0;
}

