#include <asm/current.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include "pcimem.h"

#define Cprintk(x) 
static int pcimem_open(struct inode *ip, struct file *fp);
static int pcimem_close(struct inode *ip, struct file *fp);
static int n_pcimem_exist(int vendorid, int deviceid);
static int pcimem_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int pcimem_mmap(struct file *fp, struct vm_area_struct *vp);

static int major;
static struct pci_dev *pcimem[NPCIMEM];
static int npcimem; /* # of pcimem devices found */
struct file_operations fops =
{
    NULL, /* lseek */
    NULL, /* read */
    NULL, /* write */
    NULL, /* readdir */
    NULL, /* poll */
    pcimem_ioctl, /* ioctl */
    pcimem_mmap, /* mmap */
    pcimem_open, /* open */
    NULL, /* flush */
    pcimem_close, /* close */
};

#ifdef USEDMA
static unsigned long
generic_virt_to_phys(unsigned long va, struct task_struct *process)
{
  unsigned long pa;

  pa = pte_page(
	   *pte_offset(
		       pmd_offset(
				  pgd_offset(process->mm, va),
				  va),
		       va)
	   );
  pa |= (va & (~PAGE_MASK));
  return (pa);
}
#endif
int
init_module(void)
{
    int i;

    /* register this device driver */
    major = register_chrdev(0, DEVNAME, &fops);

    for (i = 0; i < NPCIMEM; i++) {
	pcimem[i] = NULL;
    }

    /* look for pcimem device(s) */
    npcimem = n_pcimem_exist(PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9080);
    if (!npcimem) {
	return (-ENODEV);
    }

    printk(KERN_ALERT "bus: %d func: %d bar2 %08lx\n",
	   pcimem[0]->bus->number,
	   pcimem[0]->devfn,
	   pcimem[0]->base_address[2]);

    return (0);
}

void
cleanup_module(void)
{
    unregister_chrdev(major, DEVNAME);
}

static int
pcimem_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_open\n");
    return (0);
}

static int
pcimem_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "pcimem_close\n");
    return (0);
}

static int
pcimem_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    /*    printk(KERN_ALERT "pcimem_ioctl cmd: %x\n", (int) cmd);*/

    /* exec cmd with argument arg */
    switch (cmd)
    {
    case WRITE_CFG:
    {
	struct long_access *myarg = (struct long_access *)arg;
	pci_write_config_dword(pcimem[minor], myarg->addr, myarg->data);
	break;
    }
    case READ_CFG:
    {
	struct long_access *myarg = (struct long_access *)arg;
	pci_read_config_dword(pcimem[minor], myarg->addr, &myarg->data);
	break;
    }
    case MAP_BAR2_DENSE_ADDR:
    {
	break;
    }
    case DMA_MAP_LOAD:
    {
#ifdef USE_DMA	
	unsigned long va = *(unsigned long *)arg;
#endif
	unsigned long pa = 0;

	/*
        int try;
        struct task_struct *ps;
	printk(KERN_ALERT "virt: %08lx\n", va);
	printk(KERN_ALERT ">>> mm: %08lx\n", (unsigned long)current->mm);
	printk(KERN_ALERT "current: 0x%08lx\n", (unsigned long)current);
	for (ps = current, try = 0; (ps = ps->prev_task) != current && try < 3; try++) {
	  printk(KERN_ALERT "comm: %s pid: %d  mm: 0x%08lx\n",
		 ps->comm, ps->pid, (unsigned long)(ps->mm));
	}
	*/

	if (!current->mm) {
	  printk(KERN_ALERT "current->mm is missing\n");
	  return (-EINVAL);
	}
#ifdef USEDMA
		pa = generic_virt_to_phys(va, current);
#endif		
	/*
	printk(KERN_ALERT "bus: %08lx\n", pa);
	((unsigned int*)pa)[0] = 0x1234;
	*/
	pa -= PAGE_OFFSET;
	/*
	printk(KERN_ALERT "bus-offset: %08lx\n", pa);
	*/
	*(unsigned long *)arg = pa;
	break;
    }
    default:
	return (-EINVAL);
    }

    return (0);
}


static inline unsigned long pgprot_noncached(unsigned long prot)
{
#if defined(__i386__)
	/* On PPro and successors, PCD alone doesn't always mean 
	    uncached because of interactions with the MTRRs. PCD | PWT
	    means definitely uncached. */ 
	if (boot_cpu_data.x86 > 3)
		prot |= _PAGE_PCD | _PAGE_PWT;
#elif defined(__powerpc__)
	prot |= _PAGE_NO_CACHE | _PAGE_GUARDED;
#elif defined(__mc68000__)
	if (CPU_IS_020_OR_030)
		prot |= _PAGE_NOCACHE030;
	/* Use no-cache mode, serialized */
	if (CPU_IS_040_OR_060)
		prot = (prot & _CACHEMASK040) | _PAGE_NOCACHE_S;
#elif defined(__mips__)
	prot = (prot & ~_CACHE_MASK) | _CACHE_UNCACHED;
#endif

	return prot;
}

/*
 * Architectures vary in how they handle caching for addresses 
 * outside of main memory.
 */
static inline int noncached_address(unsigned long addr)
{
#if defined(__i386__)
	/* 
	 * On the PPro and successors, the MTRRs are used to set
	 * memory types for physical addresses outside main memory, 
	 * so blindly setting PCD or PWT on those pages is wrong.
	 * For Pentiums and earlier, the surround logic should disable 
	 * caching for the high addresses through the KEN pin, but
	 * we maintain the tradition of paranoia in this code.
	 */
 	return !(boot_cpu_data.x86_capability & X86_FEATURE_MTRR)
		&& addr >= __pa(high_memory);
#else
	return addr >= __pa(high_memory);
#endif
}



static int
pcimem_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = vp->vm_offset;
    unsigned long virtual = vp->vm_start;
    unsigned long physical = off;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor;

#if 0    
    printk(KERN_ALERT "pcimem_mmap off: %lx\n", off);
    printk(KERN_ALERT "pcimem_mmap vir: %lx\n", virtual);
    printk(KERN_ALERT "pcimem_mmap vsize: %lx\n", vsize);
#endif
    
    minor = MINOR(fp->f_dentry->d_inode->i_rdev);
    physical  = (pcimem[minor]->base_address[2] & PCI_BASE_ADDRESS_MEM_MASK)+off;

    if (off & (PAGE_SIZE-1)) { /* offset not aligned */
	return (-ENXIO);
    }
#if 0
	/*
	 * Accessing memory above the top the kernel knows about or
	 * through a file pointer that was marked O_SYNC will be
	 * done non-cached.
	 */
    if (noncached_address(off) || (fp->f_flags & O_SYNC)){
	pgprot_val(vp->vm_page_prot) 
	    = pgprot_noncached(pgprot_val(vp->vm_page_prot));
    }
#endif    

#if defined(__alpha__)    
#define BASE2PHYS(a) __pa(dense_mem(a)+((a)&0xffffffffUL))
    /*printk(KERN_ALERT "pcimem_mmap old physical: %lx\n", physical);*/
    physical = BASE2PHYS(physical);
    printk(KERN_ALERT "pcimem_mmap new physical: %lx\n", physical);
#endif
    ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
    /*printk(KERN_ALERT "pcimem_mmap ret: %x\n", ret);*/

    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}


static int
n_pcimem_exist(int vendorid, int deviceid)
{
    int i;
    struct pci_dev *pcimem0 = NULL;

    if (pci_present()) {
	for (i = 0; ; i++) {
	    pcimem[i] = pci_find_device(vendorid, deviceid, pcimem0);
	    pcimem0 = pcimem[i];
	    if (!pcimem[i]) {
		break;
	    }else{
		int base0, mem_size, *mem_base;
		/* Find pdev with pci_find_device() */ 
		base0 = pcimem[i]->base_address[2] & PCI_BASE_ADDRESS_MEM_MASK; 
		mem_size = 0x01000000; 
		mem_base = ioremap(base0, mem_size);
		printk(KERN_INFO "mem_base is %lx\n", (unsigned long)mem_base);
		printk(KERN_INFO "base2 is %x\n", (unsigned)base0);
	    }
	}
	if (i == 0) {
	    printk(KERN_ALERT "no pcimem found\n");
	    return (-ENODEV);
	}
	else {
	    printk(KERN_ALERT "%d pcimem(s) found\n", i);
	    return (i);
	}
    }
    else {
	printk(KERN_ALERT "pci is not supported on this machine\n");
	return (-ENODEV);
    }
}
