#include <stdio.h>
#include <stdlib.h>
#include "memmap.h"
#include "phibutil.h"

#define DEBUG (0)
#if DEBUG
#define Cfprintf fprintf
#define Cprintf printf
#else
#define Cfprintf
#define Cprintf
#endif

/* PHIB specific constants */
#define RAMSIZE (PHIBRAMWORDS)
#define RAMADDR (0x00000000)
#define LBCADDR (1<<(15-2))
#define CRST (1<<(7-2)) /* lai[CRST] read causes CB reset */
#define LRST (1<<(8-2)) /* lai[LRST] read causes HIB local reset */
#define CLR (1<<(9-2)) /* lai[CLR] read clears address counters */
#define DWREQ (1<<(6-2)) /* lai[DWREQ] read issues DMA write request */
#define QUERY_DMAW (1<<(10-2)) /* la[ISDMAW] read returns /dma_w status to ldsts */
#define QUERY_DMAR (1<<(11-2)) /* la[ISDMAR] read returns /dma_r status to ldsts */
#define QUERY_CALCSTS (0) /* la[ISDMAR] read returns /calcsts status to ldsts */
#define STSBIT (1<<15)

/* PHIB local function(s) */
 void start_read_dualportram(int devid, int size, unsigned int *buf, double ratio);
 void danger_read_dualportram(int devid, int size, double ratio);
 void read_dualportram(int devid, int size);
 void read_dualportramDMA(int devid, int size);
 void clear_cbcounter(int devid, int checkmax);
 int read_cbstatus(int devid, unsigned int query);
 int read_cbcounter(int devid, int checkmax);
 void wait_dmar_assertion(int devid);
 void wait_dmar_deassertion(int devid);
 void testrun(int devid);
 void waitfor(double maxcount);

/* PHIB local variable(s) */
static unsigned int off[MAXCLUSTERS];
static unsigned int *resbufp[MAXCLUSTERS];
static unsigned int *sharedbufp;
static unsigned long sharedpa[MAXCLUSTERS]; /* physical addres for sharedbufp */
static unsigned long resbufpa[MAXCLUSTERS]; /* physical addres for resbufp[] */
static caddr_t phib[MAXCLUSTERS];

/*
 * optimized for:
 * VT-Alpha 533MHz/256MB, PHIB/9080/32MHz, GRAPE-5/4GCs/15MHz
 */
static double dmawwait = 2.0;
static double dmarwait = 3.0;
static int nclusters = 1;

void
phib_set_nclusters(int n)
{
    if (0 < n && n <= MAXCLUSTERS)
    {
	nclusters = n;
	fprintf(stderr, "# of clusters: %d\n", nclusters);
    }
    else
    {
	nclusters = 1;
	fprintf(stderr,
		"inappropriate # of clusters (=%d)\nuse default (=1).",
		n);
    }
}

int
phib_get_nclusters(void)
{
    return (nclusters);
}

void
phib_optimizeDMA(double ww, double rw)
{
    dmawwait = ww;
    dmarwait = rw;
}

/* interfaces for single cluster */
unsigned int *
phib_open(void)
{
    return (phib_openMC(0));
}

void
phib_close(void)
{
    phib_closeMC(0);
}

void
phib_piow(unsigned int addr, unsigned int cmd)
{
    phib_piowMC(0, addr, cmd);
}

void
phib_dmar(int size, unsigned int *buf)
{
    phib_dmarMC(0, size, buf);
}

void
phib_dmaw(int size, unsigned int *buf)
{
    phib_dmawMC(0, size, buf);
}

int
phib_calcsts(void)
{
    return (phib_calcstsMC(0));
}

void
phib_wait(int nclk)
{
    phib_waitMC(0, nclk);
}

void
phib_pioconfw(unsigned int addr, unsigned int cmd)
{
    phib_pioconfwMC(0, addr, cmd);
}

caddr_t
phib_mapped_addrMC(int devid)
{
    return (phib[devid]);
}

/*
 * open PHIB
 * returns pointer to the data buffer
 */
unsigned int *
phib_openMC(int devid)
{
    unsigned int *ret;

    ret = phib_open_notestMC(devid);
    testrun(devid);
    return (ret);
}

unsigned int *
phib_open_notestMC(int devid)
{
    int ic;
    unsigned int val;
    static int firstcall = 1;
    fprintf(stderr,"open_notest called %d\n", devid);

    if (nclusters < devid+1)
    {
	fprintf(stderr,
		"phib_openMC(): too large devid(= %d).\n",
		devid);
	fprintf(stderr, "set # of clusters first.\n");
	exit(1);
    }

    while (-1 == (int)(phib[devid] = TBopen(devid)))
    {
	sleep(1);
    }

    if (firstcall)
    {
	firstcall = 0;

	sharedbufp = 
	    (unsigned int *)calloc(sizeof(unsigned int), RAMSIZE*(MAXCLUSTERS+1));
	if (sharedbufp == NULL)
	{
	    fprintf(stderr, "phib_openMC(): calloc() failed\n");
	    exit (2);
	}
	for (ic = 0; ic < nclusters; ic++)
	{
	    off[ic] = 0;
	    sharedpa[ic] = NULL;
	    resbufp[ic] = sharedbufp + RAMSIZE*(ic+1);
	}
    }
    if (sharedpa[devid] == NULL)
    {
	sharedpa[devid] =
	    TBdmaMapLoad(devid, (caddr_t)sharedbufp,
			 RAMSIZE*(MAXCLUSTERS+1));
	fprintf(stderr, "size: %d\n", RAMSIZE*(MAXCLUSTERS+1));
	fprintf(stderr, "phib%d: sharedpa %08x\n",
		devid, sharedpa[devid]);
    }
    resbufpa[devid] = sharedpa[devid] + RAMSIZE*(devid+1);
    resbufpa[devid] = sharedpa[devid];

    val = TBregRead(devid, 0x6c);
    TBregWrite(devid, 0x6c, val & 0xfffffffc); /* mem read multiple */
    TBregWrite(devid, 0x84, resbufpa[devid]);

    Cprintf("la0: 0x%0x\n", resbufp[devid]);
    Cfprintf(stderr, "phib[%d] pa0: 0x%0x\n",
	    devid, sharedpa[devid]);
    Cfprintf(stderr, "phib[%d] pa1: 0x%0x\n",
	    devid, resbufpa[devid]);
    fprintf(stderr,"open_notest end %d\n", devid);

/*
    return (unsigned int *)resbufp[devid];
    */
    return (unsigned int *)sharedbufp;
}

/* close PHIB */
void
phib_closeMC(int devid)
{
    TBterm(devid);
}

/* wait for more than nclk*30ns */
void
phib_waitMC(int devid, int nclk)
{
    int j;
    int nmax = 1000;
    int dummyoff = 1000;

    while (nclk > 0)
    {
	if (nclk < nmax)
	{
	    nmax = nclk;
	}
	for (j = 0; j < nmax; j++)
	{
	    *(((unsigned int *)phib[devid])+j+dummyoff) = 0;
	}
	nclk -= nmax;
    }
}

/* send 'cmd' to PG with PIO write operation */
void
phib_piowMC(int devid, unsigned int addr, unsigned int cmd)
{
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_piowMC: open phib%d first.\n", devid);
	exit (2);
    }

    clear_cbcounter(devid, 0);
    TBmemWriteInt(off[devid], addr);
    off[devid]++;
    TBmemWriteInt(off[devid], cmd);
    off[devid]++;
    TBmemWriteInt(LBCADDR, off[devid]);
}

/* wait until PIOW finishes */
void
phib_pioconfwMC(int devid, unsigned int addr, unsigned int cmd)
{
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_pioconfwMC: open phib%d first.\n", devid);
	exit (2);
    }

    clear_cbcounter(devid, 0);
    TBmemWriteInt(off[devid], addr);
    off[devid]++;
    TBmemWriteInt(off[devid], cmd);
    off[devid]++;
    TBmemWriteInt(LBCADDR, off[devid]);
    while (read_cbcounter(devid, 0) != 2)
    {
    }
}

void
phib_add_rcntMC(int devid, int cnt)
{
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_add_rcntMC: open phib%d first.\n", devid);
	exit (2);
    }

    off[devid] += cnt;

    *(((u_int *)mp)+(LBCADDR)) = off[devid];
/*
    TBmemWriteInt(LBCADDR, off[devid]);
    */
}

void
phib_wait_dmar_assertionMC(int devid)
{
    wait_dmar_assertion(devid);
}

void
phib_wait_dmar_deassertionMC(int devid)
{
    wait_dmar_deassertion(devid);
}

/* send 'size' words from 'buf' to PG
   with DMA read operation */
void
phib_dmarMC(int devid, int size, unsigned int *buf)
{
    phib_dmar_nowaitMC(devid, size, buf);
    wait_dmar_deassertion(devid);
}

void
phib_broadcast(int size, unsigned int *buf)
{
    int i, ic;
    caddr_t mp;

    for (ic = 0; ic < nclusters; ic++)
    {
	if (off[ic]+size > RAMSIZE)
	{
	    fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	    fprintf(stderr, "off: %d  size: %d\n", off[ic], size);
	    exit(2);
	}
	if (off[ic] != off[0])
	{
	    fprintf(stderr, "off[%d]: %d not sync with off[0]: %d\n",
		    off[ic], off[0]);
	    exit(2);
	}

	if (sharedbufp > buf ||
	    sharedbufp+RAMSIZE < buf+off[ic]+size)
	{
	    TBregWrite(ic, 0x84, sharedpa[ic]);
	    for (i = 0; i < size; i++)
	    {
		sharedbufp[i] = buf[i];
	    }
	}
	else
	{
	    /* start address in PCI space */
	    TBregWrite(ic, 0x84, (unsigned int*)sharedpa[ic]+(buf - sharedbufp));
	}

	TBregWrite(ic, 0x88, off[ic]*4); /* start address in local space */
	TBregWrite(ic, 0x8c, size*4); /* size in byte */
	TBregWrite(ic, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    }

    for (ic = 0; ic < nclusters; ic++)
    {
	wait_dmar_assertion(ic);
	TBregWrite(ic, 0xa8, 0x00000003); /* start DMA */

#define DMAPAR (0)

#if DMAPAR
    }
#endif

    waitfor(dmarwait*size);

#if DMAPAR
    for (ic = 0; ic < nclusters; ic++)
    {
#endif
	mp = phib[ic];
	if (!mp)
	{
	    fprintf(stderr, "phib_broadcast(): open phib%d first.\n", ic);
	    exit(2);
	}
	while (!(TBregRead(ic, 0xa8) & 0x10))
	{
	}
	off[ic] += size;
	TBmemWriteInt(LBCADDR, off[ic]);
    }
}

void
phib_dmar_nowaitMC(int devid, int size, unsigned int *buf)
{
    int i;
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_dmar_nowaitMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	fprintf(stderr, "off: %d  size: %d\n", off[devid], size);
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+off[devid]+size)
    {
	TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }
    else
    {
	/* start address in PCI space */
	TBregWrite(devid, 0x84,
		   (unsigned int *)resbufpa[devid]+
		   (buf - resbufp[devid]));
    }

    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    wait_dmar_assertion(devid);
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
    off[devid] += size;
    TBmemWriteInt(LBCADDR, off[devid]);
}

void
phib_dmar_rawMC(int devid, int size, unsigned int *buf)
{
    int i;
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_dmar_rawMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmar_rawMC(): size too large.\n");
	fprintf(stderr, "size: %d\n", size);
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+size)
    {
	TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }
    else
    {
	/* start address in PCI space */

	Cfprintf(stderr, "buf: %08x   resbufp[%d]: %08x\n",
		 buf, devid, resbufp[devid]);

	TBregWrite(devid, 0x84,
		   (unsigned int *)resbufpa[devid]+
		   (buf - (unsigned int *)resbufp[devid]));
    }

    TBregWrite(devid, 0x88, 0); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

void
phib_dmar_raw_multiple(int size)
{
    int ic;

    for (ic = 0; ic < nclusters; ic++)
    {
	if (size > RAMSIZE)
	{
	    fprintf(stderr, "phib_dmarMC(): size too large.\n");
	    fprintf(stderr, "size: %d\n", size);
	    exit(2);
	}

	Cfprintf(stderr, "phib_dmar_raw_multiple() pa[%d]: %08x\n",
		 ic, sharedpa[ic]);

	Cfprintf(stderr, "phib_dmar_raw_multiple() phib%d: sharedpa %08x\n",
		 ic, sharedpa[ic]);

	TBregWrite(ic, 0x84, sharedpa[ic]); /* start address in PCI space */
	TBregWrite(ic, 0x88, 0); /* start address in local space */
	TBregWrite(ic, 0x8c, size*4); /* size in byte */
	TBregWrite(ic, 0x90, 0x00000000); /* direction (0:PCI --> local) */
	TBregWrite(ic, 0xa8, 0x00000003); /* start DMA */
    }

    waitfor(dmarwait*size);

    for (ic = 0; ic < nclusters; ic++)
    {
	while (!(TBregRead(ic, 0xa8) & 0x10))
	{
	}
    }
    for (ic = 0; ic < nclusters; ic++)
    {
	TBregWrite(ic, 0x84, resbufpa[ic]);
    }
}

/* receive 'size' words from PG to 'buf'
   with DMA write operation */
void
phib_dmawMC(int devid, int size, unsigned int *buf)
{
    int i;
    int rcnt = 0;

    read_dualportram(devid, size);

    /* copy contents of 'resbufp[devid]' to 'buf' buffer
       specified by the user of this library */
    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    buf[i] = ((unsigned int *)resbufp[devid])[i];
	}
    }
}

void
phib_start_dmawMC(int devid, int size, unsigned int *buf, double ratio)
{
    start_read_dualportram(devid, size, buf, ratio);
}

void
phib_finish_dmawMC(int devid, int len)
{
    waitfor(dmawwait*len);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

void
phib_danger_dmawMC(int devid, int size, unsigned int *buf, double ratio)
{
    danger_read_dualportram(devid, size, ratio);
}

void
phib_waitfor(int size)
{
    waitfor(dmawwait*size);
}


int
phib_calcstsMC(int devid)
{
    unsigned int val;
	
    val = read_cbstatus(devid, LBCADDR|QUERY_CALCSTS);
    MB;

    Cfprintf(stderr, "val: %d\n", val);

    return (val);
}

/*
 * receive data from dual-port ram to 'resbufp[devid]'.
 */
#define NTRYMAX (10)

static void
start_read_dualportram(int devid, int size, unsigned int *buf, double ratio)
{
    register int rcnt = 0;
    register int dmastart;
	
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "danger_read_dualportram(): offset+size too large.\n");
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+off[devid]+size)
    {
	fprintf(stderr, "start_read_dualportram(): inappropriate DMAW range\n");
	fprintf(stderr, "off: %d  size: %d\n", off[devid], size);
	fprintf(stderr, "resbufp[%d]: %08x buf: %08x\n",
		devid, resbufp[devid], buf);
	exit(1);
    }
    else
    {
	/* start address in PCI space */
	TBregWrite(devid, 0x84,
		   (unsigned int *)resbufpa[devid]+
		   (buf - (unsigned int *)resbufp[devid]));
    }
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */

    rcnt = read_cbcounter(devid, 0);
    dmastart = size*ratio+off[devid];
    while (rcnt < dmastart)
    {
	rcnt = read_cbcounter(devid, 0);
    }

    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    off[devid] = off[devid] + size;
}



static void
danger_read_dualportram(int devid, int size, double ratio)
{
    register int rcnt = 0;
    register int dmastart;
	
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "danger_read_dualportram(): offset+size too large.\n");
	exit(2);
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */

    rcnt = read_cbcounter(devid, 0);
    dmastart = size*ratio+off[devid];
    while (rcnt < dmastart)
    {
	rcnt = read_cbcounter(devid, 0);
    }

    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    waitfor(dmawwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }

    off[devid] = off[devid] + size;
}

static void
read_dualportram(int devid, int size)
{
    register int rcnt = 0;
    int i, ntry = 0;
    int rcntfin, rcnt2, rcnt0 = 0;

    /* wait until all data are written to dual-port ram on PHIB */
    rcntfin = off[devid] + size;
    while (rcnt != rcntfin)
    {
	rcnt = read_cbcounter(devid, 2);

/*
	if (rcnt != rcnt0)
	{
	    fprintf(stderr, "rcnt: %d  rcntfin: %d\n", rcnt, rcntfin);
	    rcnt0 = rcnt;
	}
	*/
    }
    Cfprintf(stderr, "rcnt reached rcntfin\n");

    do
    {
	if (ntry)
	{
	    rcnt = read_cbcounter(devid, 2);
	}
	read_dualportramDMA(devid, size);
	ntry++;
	rcnt2 = read_cbcounter(devid, 0);
    } while (rcnt != rcnt2 && ntry < NTRYMAX);

    if (rcnt != rcnt2)
    {
	fprintf(stderr, "read_dualportram(): cannot correct. abort.\n");
	exit (1);
    }
    off[devid] = off[devid] + size;
}

static void
read_dualportramDMA(int devid, int size)
{
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "read_dualportramDMA(): offset+size too large.\n");
	exit(2);
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    waitfor(dmawwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

void
phib_dmartestMC(int devid, int size, unsigned int *buf)
{
    int i;
    caddr_t mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_pioconfwMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmarMC(): size too large.\n");
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, 0); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    wait_dmar_assertion(devid);
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
    off[devid] += size;
    TBmemWriteInt(LBCADDR, off[devid]);
    wait_dmar_deassertion(devid);
}

void
phib_dmawtestMC(int devid, int size, unsigned int *buf)
{
    int i;
    int rcnt = 0;

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmawMC(): size too large.\n");
	exit (1);
    }

    off[devid] = 0;
    read_dualportramDMA(devid, size);

    /* copy contents of 'resbufp[devid]' to 'buf' buffer
       specified by the user of this library */
    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    buf[i] = ((unsigned int *)resbufp[devid])[i];
	}
    }
}

/*************  functions below should not be modified  *************/

static void
clear_cbcounter(int devid, int checkmax)
{
    off[devid] = RAMADDR;
    TBmemRead(devid, LBCADDR|CLR);
}

static int
read_cbstatus(int devid, unsigned int query)
{
    int data;

    data = TBmemRead(devid, query);
    data = STSBIT & data;

    if (data > 0)
    {
	data = 1;
    }
    return (data);
}

static int
read_cbcounter(int devid, int checkmax)
{
    int check;
    int data, data1;

    data = TBmemRead(devid, LBCADDR);
    MB;
    data = 0x00000fff & data;
    for (check = 0; check < checkmax; check++)
    {
	data1 = TBmemRead(devid, LBCADDR);
	MB;
	data1 = 0x00000fff & data1;
	if (data != data1)
	{
	    /* counter read error */
	    Cprintf("read_cbcounter data1: 0x%08x\n", data1);
	    data = -1;
	}
    }
    return (data);
}

/* wait until CB issues DMA-read request */
static void
wait_dmar_assertion(int devid)
{		
    Cfprintf(stderr, "waiting for assertion...\n");
			
    while (read_cbstatus(devid, LBCADDR|QUERY_DMAR) > 0)
    {
	MB;
	Cprintf("#a\n");
    }
    Cfprintf(stderr, "asserted\n");
}

static void
wait_dmar_deassertion(int devid)
{		
    Cfprintf(stderr, "waiting for deassertion...\n");

    Cprintf("## before counter: %d\n", read_cbcounter(devid, 1));
    while (read_cbstatus(devid, LBCADDR | QUERY_DMAR) == 0)
    {
	Cprintf("+++ counter: %d\n", read_cbcounter(devid, 1));
	MB;
	Cprintf("#d\n");
    }

    Cprintf("## after counter: %d\n", read_cbcounter(devid, 1));
    Cfprintf(stderr, "deasserted\n");
}

static void
testrun(int devid)
{
    int j;
    int err = 0;
    int *buf;

    buf = (int *)calloc(sizeof(unsigned int), RAMSIZE);
    Cfprintf(stderr, "sharedbufp: %08x  resbufp[%d]: %08x\n",
	    sharedbufp, devid, resbufp[devid]);
    Cfprintf(stderr, "sharedpa[%d]: %08x\n", devid, sharedpa[devid]);
    Cfprintf(stderr, "resbufpa[%d]: %08x\n", devid, resbufpa[devid]);
    Cfprintf(stderr, "RAMSIZE: %08x words\n", RAMSIZE);
    Cfprintf(stderr, "DMA size: %08x bytes\n", RAMSIZE*sizeof(unsigned int));

    resbufp[devid] = sharedbufp+0x0;
/*
    resbufpa[devid] = TBdmaMapLoad(devid, (caddr_t)resbufp[devid]);
    */

    TBregWrite(devid, 0x80, 0x000001c3);
/*
    TBregWrite(devid, 0x84, resbufpa[devid]);
*/
    TBregWrite(devid, 0x84, sharedpa[devid]+0x0); /* start address in PCI space */

    TBregWrite(devid, 0x88, 0x00000000);
    TBregWrite(devid, 0x8c, RAMSIZE*sizeof(unsigned int));

    for (j = 0; j < RAMSIZE; j++)
    {
	buf[j] = j*4;
	*((int *)resbufp[devid]+j) = 0xf;
    }
    memcpy(resbufp[devid], buf, RAMSIZE*sizeof(int));

    /* write */
    TBregWrite(devid, 0x90, 0x00000000);
    TBregWrite(devid, 0xa8, 0x00000003);

    Cfprintf(stderr, "testrun 0\n");
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
    Cfprintf(stderr, "testrun 1\n");

    memset(resbufp[devid], 0, RAMSIZE*sizeof(int));

    /* read */
    TBregWrite(devid, 0x90, 0x00000008);
    TBregWrite(devid, 0xa8, 0x00000003);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }

    for (j = 0; j < RAMSIZE; j++)
    {
	if (buf[j] != resbufp[devid][j])
	{
	    fprintf(stderr, "buf[%0x]: 0x%0x  rbp[%0x]: 0x%0x\n",
		    j, buf[j], j, resbufp[devid][j]);
	    err = 1;
	}
#if 0
	else
	{
	    fprintf(stderr, "buf[%0x]: 0x%0x  rbp[%0x]: 0x%0x  OK\n",
		    j, buf[j], j, resbufp[devid][j]);
	}
#endif
    }
    if (err)
    {
	fprintf(stderr, "tried to load DMA map but failed.\n");
	exit(1);
    }
    free(buf);
}

static void
waitfor(double maxcount)
{
    int i, j;
    int jmax = maxcount;

    for (j = 0; j < maxcount; j++)
    {
	for (i = 0; i < 10; i++)
	{
	}
    }
}
