/*
 *  GRIO.c  : I/O routines for GRAPE-1
 *
 *  Jun Makino 89/09/22 Ver. 0.00 (created)
 *             89/09/26 Ver. 0.01
 *                      Does not overwrite existing files
 *             89/10/02 Ver. 0.02
 *                      Modified for UNIX system
 *                      NOGRAPHICS flag for systems with which on-the-fly
 *                      graphics is not available or slow.
 *             90/11/13 Ver. 0.10
 *                      NEMO filestruct output support (-DNEMOOUT)
 *
 *  input file format : atos format (nemo)
 *  output file format : binary raw file (4 byte int, REAL)
 *
 *
 */
#ifndef REAL
#define REAL float
#define SHORTREAL
#endif
#ifndef NEMOOUT
#include <stdio.h>
#else
#include <stdinc.h>
#endif

#ifdef MSC
#   define READBINARY "rb"
#   define WRITEBINAEY "wb"
#else
#   define READBINARY "r"
#   define WRITEBINARY "w"
#endif

static FILE * in_snap_file = NULL;
#ifndef NEMOOUT
static FILE * out_snap_file = NULL;
#else
static stream out_snap_file = NULL;
#endif
static FILE * out_raster_file = NULL;

int set_input_file(parmfile)
    FILE * parmfile;
{
	char inname[200];
	int iret;
	fprintf(stderr,
	       "Enter in_snap file name\n");
	fscanf(parmfile,"%s", inname);
	in_snap_file = fopen(inname, "r");
	if(in_snap_file == NULL){
		printf("Can't open input snap file %s\n", inname);
		return 1;
	}
	printf("input snap = %s\n",inname);
	return 0;
}

int file_exist(name)
char name[];
{
	FILE * fptr;
	if((fptr=fopen(name,"r"))!=NULL){
		close(fptr);
		return 1;
	}else{
#ifdef MSC
		clearerr(fptr);
#endif
		return 0;
	}
}
void output_file_check_and_exit(name)
char name[];
{
	if(file_exist(name)){
		fprintf(stderr,"output file %s exists\n",name);
		exit(1);
	}
}
		
int set_output_snap_file(parmfile)
    FILE * parmfile;
{
	char outname[200];
	int iret;
	fprintf(stderr,"Enter out_snap file name( ~ for no output)\n");
	fscanf(parmfile,"%s", outname);
	if(outname[0]&& outname[0] != '~'){
#ifndef NEMOOUT	    
		output_file_check_and_exit(outname);
		out_snap_file = fopen(outname, WRITEBINARY);
		if(out_snap_file == NULL){
			printf("Can't create output snap file %s\n", outname);
			return 2;
		}
#else
		out_snap_file = stropen(outname,"w");
#endif		
	}else{
		out_snap_file = NULL;
	}
	printf("output snap = %s\n", outname);
	return 0;
}

int set_input_binary_file(file)
FILE ** file;
{
	char inname[200];
	int iret;
	fprintf(stderr,
	       "Enter in_snap file name (binary) :");
	scanf("%s", inname);
	in_snap_file = fopen(inname, READBINARY);
	if(in_snap_file == NULL){
		printf("Can't open input snap file %s\n", inname);
		return 1;
	}
	*file = in_snap_file;
	printf("input snap = %s\n",inname);
	return 0;
}

int set_out_raster_file()
{
	char outname[200];
	int iret;
	fprintf(stderr,
	       "Enter out_raster file name (binary) :");
	scanf("%s", outname);
	if(outname[0] == 0){
		out_raster_file = NULL;
		return 0;
	}
	output_file_check_and_exit(outname);
	out_raster_file = fopen(outname, WRITEBINARY);
	if(out_raster_file == NULL){
		printf("Can't open output_raster file %s\n", outname);
		return 1;
	}
	printf("raster = %s\n",outname);
	return 0;
}

int write_raster(x1,y1,x2,y2,work,cmode)
int x1,y1,x2,y2;
char work[];
int cmode;
{       
	int xx1,yy1,xx2,yy2,ccmode;
#ifdef NOGRAPHICS
	return 1;
#else
	xx1=x1/8;
	xx2=x2/8;
	yy1=y1/16;
	yy2=y2/16;
	ccmode=cmode;
	if(out_raster_file != NULL){
		printf("(write raster) %d %d %d %d %d\n", 
			xx1,yy1,xx2,yy2,ccmode);
		window_getput(xx1,yy1,xx2,yy2,work,0,&ccmode);
		return window_put_file(out_raster_file,
			&xx1,&yy1,&xx2,&yy2,work,&ccmode);
	}else{
		return 0;
	}
#endif
}
	
int read_snap(n,t,m,x,v, nmax)
int * n;	/* output	number of particles */
REAL * t;	/* output	system time */
REAL m[];	/* output	mass data */
REAL x[][3];	/* output	position data */
REAL v[][3];	/* output	velocity data */
int nmax;	/* input	max of particle number */
{
	int ndim, i, k;
	fscanf(in_snap_file,"%d", n);
	if(*n > nmax){
		printf("particle number %d exceeds max (%d)\n",*n, nmax);
		return 1;
	}
	fscanf(in_snap_file,"%d", &ndim);
	if(ndim != 3){
		printf("dimension number %d not suppoeted (should be %d)\n",
		ndim, 3);
		return 1;
	}
#ifdef SHORTREAL
#define FORM1 "%f"	
#define FORM3 "%f%f%f"
#else
#ifdef sun4
#define FORM1 "%F"	
#define FORM3 "%F%F%F"
#else
#define FORM1 "%lf"	
#define FORM3 "%lf%lf%lf"
#endif	
#endif	
	fscanf(in_snap_file,FORM1, t);
	for(i=0; i<*n; i++){
		fscanf(in_snap_file,FORM1,&m[i]);
	}
	for(i=0; i<*n; i++){
		fscanf(in_snap_file,FORM3,&x[i][0],&x[i][1],&x[i][2]);
	}
	for(i=0; i<*n; i++){
		fscanf(in_snap_file,FORM3,&v[i][0],&v[i][1],&v[i][2]);
	}
	return 0;
}
int out_snap_binary(n,t,m,x,v,pot,pflag)
int n;		/* input	number of particles */
REAL t;	/* input	system time */
REAL m[];	/* input	mass data */
REAL x[][3];	/* input	position data */
REAL v[][3];	/* input	velocity data */
REAL pot[];	/* input	potential data */
int pflag;	/* input        flag to output potential */
{
    
    long int nl, ndiml;
    int iword;
    REAL tdum;
#if NEMOOUT
    int iret;
    if(out_snap_file == NULL){
	return 0;
    }
    if(pflag){
	iret = nemo_snapout(n,t,m,x,v,pot,out_snap_file);
    }else{
	iret = nemo_snapout(n,t,m,x,v,NULL,out_snap_file);
    }
    fflush(out_snap_file);
    return iret;
#else    
    if(out_snap_file == NULL){
	return 0;
    }
    tdum = t;
    nl = n;
    ndiml = 3;
    if(fwrite((char *) &nl,sizeof(nl),1,out_snap_file) != 1){
	puts("(out_snap_binary) write n failed");
	return 1;
    }
    if(fwrite((char *) &ndiml,sizeof(ndiml),1,out_snap_file) != 1){
	puts("(out_snap_binary) write ndim failed");
	return 1;
    }
    if(fwrite((char *) &tdum,sizeof(tdum),1,out_snap_file) != 1){
	puts("(out_snap_binary) write t failed");
	return 1;
    }
    if(fwrite((char *) m,sizeof(m[0]),n,out_snap_file) != n){
	puts("(out_snap_binary) write m failed");
	return 1;
    }
    if(fwrite((char *) x,sizeof(x[0][0])*3,n,out_snap_file) != n){
	puts("(out_snap_binary) write x failed");
	return 1;
    }
    if(fwrite((char *) v,sizeof(v[0][0])*3,n,out_snap_file) != n){
	puts("(out_snap_binary) write x failed");
	return 1;
    }
    if(pflag){
	fprintf(stderr,"output potential --- \n");
	if(fwrite((char *) pot,sizeof(pot[0]),n,out_snap_file) != n){
	    puts("(out_snap_binary) write pot failed");
	    return 1;
	}
    }
    fflush(out_snap_file);
    return 0;
#endif    
}

int read_snap_binary(n,t,m,x,v,nmax)
int * n;	/* output	number of particles */
REAL * t;	/* output	system time */
REAL m[];	/* output	mass data */
REAL x[][3];	/* output	position data */
REAL v[][3];	/* output	velocity data */
int nmax;	/* input	limit size of the array */
{
	
	long int nl, ndiml;
	int iword, ndim, ntmp;
	REAL tdum;
	if(in_snap_file == NULL){
		return 0;
	}
	if(fread((char *) &nl,sizeof(nl),1,in_snap_file) != 1){
		puts("(read_snap_binary) read n failed");
		return 1;
	}
	*n = nl;
	ntmp = nl;
	if(*n > nmax){
		printf("(read_snap_binary) n=%ld too large (>%d)\n",*n, nmax);
		return 1;
	}
		
	if(fread((char *) &ndiml,sizeof(ndiml),1,in_snap_file) != 1){
		puts("(read_snap_binary) read ndim failed");
		return 1;
	}
	if(ndiml != 3){
		printf("(read_snap_binary) ndim=%ld illegal\n",ndiml);
	}
	ndim = ndiml;
	if(fread((char *) &tdum,sizeof(tdum),1,in_snap_file) != 1){
		puts("(read_snap_binary) read t failed");
		return 1;
	}
	*t = tdum;
	if(fread((char *) m,sizeof(m[0]),ntmp,in_snap_file) != ntmp){
		puts("(read_snap_binary) read m failed");
		return 1;
	}
	if(fread((char *) x,sizeof(x[0][0])*3,ntmp,in_snap_file) != ntmp){
		puts("(read_snap_binary) read x failed");
		return 1;
	}
	if(fread((char *) v,sizeof(v[0][0])*3,ntmp,in_snap_file) != ntmp){
		puts("(read_snap_binary) read x failed");
		return 1;
	}

	return 0;
}


