/*
 * grape.c : a simple N-body program to be used with the GRAPE-1
 *           Pipelined N-body engine
 *
 * J. Makino   89/09/16   Ver. 1.00 (created)
 *                        Workbench version with exact force calculation.
 *             89/09/26   Ver. 1.01 
 *                        Running with GRAPE-1 hardware (since 9/22?)
 *                        Optional input for cold collapes calculation
 *	       89/10/02   Ver. 1.02
 *			  Modified for UNIX system, add CPP flags
 *                        EXACT_CALCULATION : not use grape hardware
 *             89/10/12   Ver. 1.03
 *                        Initial power law density supported
 *             90/02/18   Ver. 1.04
 *                        Outer plummer potential field added
 *             90/11/13   Ver. 1.10
 *                        GRAPE-1A interface added (-DGRAPE1A)
 *                        NEMO filestruct interface added (-DNEMOOUT)
 * 
 *
 */
#define LFFORMAT
#ifdef ibm
#define LFFORMAT
#else
#ifdef DEC
#define LFFORMAT
#endif
#endif    
#ifdef GRAPE1A
# undef GRAPE1A
# define GRAPE1A 1
#else
# define GRAPE1A 0
#endif
#ifdef GRAPE3
# undef GRAPE3
# define GRAPE3 1
#else
# define GRAPE3 0
#endif
#ifdef GRAPE2A
#define GRAPE2A 1
#else
#define GRAPE2A 0
#endif
#ifdef HARP3
#define HARP3 1
#else
#define HARP3 0
#endif
#ifdef GRAPE6
#define GRAPE6 1
#else
#define GRAPE6 0
#endif
#if !(GRAPE3 | GRAPE1A|GRAPE2A|GRAPE2|HARP3|GRAPE6)
#   define GRAPE_1 1
#else
#   define GRAPE_1 0
#endif
#ifndef REAL
#define SHORTREAL
#ifdef SHORTREAL
#   define REAL float
#else
#   define REAL double
#endif
#endif

#ifdef MSC
#    define SHORTINT
#else
#    define LONGINT
#endif
#ifndef NOSHARE
# define NOSHARE 0
#else
# undef NOSHARE
# define NOSHARE 1
#endif
#ifndef BIG
#if GRAPE2A
#define NMAX 16384
#else
#define NMAX 1500000
#endif
#else
#define NMAX 1500000
#endif
#define NDIM 3
#include <math.h>
#ifndef NEMOOUT
#include <stdio.h>
#ifdef DEC
#include <stdlib.h>
#endif
#else
#include <stdinc.h>
#include <getparam.h>
#include <vectmath.h>
#include <filestruct.h>


double pow();
double sqrt();
/**************** COMMAND LINE PARAMETERS **********************/

char *defv[] = {                /* DEFAULT INPUT PARAMETERS */
    "in=-",                   /* input file name */
#ifdef YAPP_SV
    "frame=",			/* screendump file name */
#endif    
    "VERSION=1.10",		/* ID */
    NULL
};

#endif

static double jerk[NMAX][3];

void print_accel(int n, double acc[][3], double p[])
{
    int i, k;
    puts("calcualted gravity");
    for(i=0; i<n; i++){
	for(k=0; k<3; k++){
	    printf(" %15.8e", acc[i][k]);
	}
	printf(" %15.8e", p[i]);
	printf("\n");
    }
}


int check_stopfile();
int check_stopfile()
{
/* if file exists, return 0
   else return -1 */
#define DIRNAME "/tmp"
#ifdef SUNOS    

#include <dirent.h>
    DIR * dirp;
  struct dirent * dp;
  char * name = "stop_grape3a";
  dirp = opendir(DIRNAME);
  if(dirp == NULL){
    fprintf(stderr,"Cannot open %s, exiting\n",DIRNAME);
    exit(1);
  }
  for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)){
#ifdef INTERNAL_OUT_FULL
    fprintf(stderr,"Name : %s\n", dp->d_name);
#endif
    if (!strcmp(dp->d_name, name)) {
      closedir (dirp);
#if INTERNAL_OUT
      fprintf(stderr,"Entry %s found\n",name);
#endif
      free(dirp);
      return 0;
    }
  }
  closedir (dirp);
#if INTERNAL_OUT
    fprintf(stderr,"Entry %s not found\n",name);
#endif
    free(dirp);
  return -1;
#else /* NON SUN OS */
    return -1;
#endif    
}

static REAL tsys;
static REAL potchange = 0.0;
void print_cpusec()
{
	double cpu;
	second(&cpu);
	
	printf("CPU sec.= %11.2f\n", cpu);
}

void print_cpusec_stderr()
{
	double cpu;
	second(&cpu);
	fprintf(stderr, "CPU sec.= %11.2f\n", cpu);
}

#ifdef OUTER_POTENTIAL
void add_outer_force(n,x,a)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* input	position array*/
    REAL a[][NDIM];	/* output	calculated acceleration */
{
    int i, j, k;
    REAL f[NDIM];

/*    puts("(accel_accurate) entered");*/
    for(i=0; i<n; i++){
	outer_force(f,x[i],tsys);
    	for(k=0;k<NDIM;k++){
    	    a[i][k] += f[k];
    	}
    }
}
#else
#define add_outer_force(n,x,a) ;
#endif
    
void accel_accurate(n,x,m,eps2,a)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* input	position array*/
    REAL m[];		/* input	particle mass */
    REAL eps2;		/* input	softening parameter */
    REAL a[][NDIM];	/* output	calculated acceleration */
{
    int i, j, k;
    REAL dx, dy, dz, r2, r3inv, fi, fj;

/*    puts("(accel_accurate) entered");*/
    for(i=0; i<n; i++){
    	for(k=0;k<NDIM;k++){
    	    a[i][k] = 0.0;
    	}
    }
    for(i=0; i<n-1; i++){
    	for(j=i+1; j<n; j++){
    	    dx = x[i][0] - x[j][0];
    	    dy = x[i][1] - x[j][1];
    	    dz = x[i][2] - x[j][2];
    	    r2 = dx*dx + dy*dy + dz*dz + eps2;
    	    r3inv = 1.0/(sqrt(r2)*r2);
    	    fi = -m[j]*r3inv;
    	    fj =  m[i]*r3inv;
    	    a[i][0] += fi*dx;
    	    a[i][1] += fi*dy;
    	    a[i][2] += fi*dz;
    	    a[j][0] += fj*dx;
    	    a[j][1] += fj*dy;
    	    a[j][2] += fj*dz;
    	}
    }
}

void push_velocity_half(n,v,a,dt)
    int  n;		/* input	number of particles */
    register REAL v[][NDIM];	/* in/out	velocity array */
    register REAL a[][NDIM];	/* input	acceleration array */
    REAL dt;		/* input	timestep */
{
    register int i, k;
    register REAL dthalf;
    
    dthalf = dt*0.5;
    for(i=0; i<n; i++){
#if 0	
    	for(k=0; k<NDIM; k++){
    	    v[i][k] += dthalf*a[i][k];
    	}
#else
	v[i][0] += dthalf*a[i][0];
	v[i][1] += dthalf*a[i][1];
	v[i][2] += dthalf*a[i][2];
#endif	
    }
}

void push_velocity_full(n,v,a,dt)
    int  n;		/* input	number of particles */
    register REAL v[][NDIM];	/* in/out	velocity array */
    register REAL a[][NDIM];	/* input	acceleration array */
    register REAL dt;		/* input	timestep */
{
    register int i, k;
    for(i=0; i<n; i++){
	v[i][0] += dt*a[i][0];
	v[i][1] += dt*a[i][1];
	v[i][2] += dt*a[i][2];
    }
}

void push_position_full(n,x,v,dt)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* in/out	position array */
    REAL v[][NDIM];	/* input	velocity array */
    register REAL dt;		/* input	timestep */
{
    register int i, k;

    for(i=0; i<n; i++){
#if 0
    	for(k=0; k<NDIM; k++){
    	    x[i][k] += dt*v[i][k];
    	}
#else
	x[i][0] += dt*v[i][0];
	x[i][1] += dt*v[i][1];
	x[i][2] += dt*v[i][2];
#endif	
    }
}

void print_array(n,x)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* input	array to print*/
{
    int i,k;
    for(i=0; i<n; i++){
    	printf("%4d ", i);
    	for(k=0; k<NDIM; k++){
    	    printf(" %18.9f", x[i][k]);
    	}
    	printf("\n");
    }
}

#ifdef PCMODE
static REAL aold[NMAX][NDIM];

void save_old_acc(a0,a,n)
    REAL a0[][NDIM];
    REAL a[][NDIM];
    int n;
{
    register int i,k;
    for(i=0;i<n;i++){
	for(k=0;k<NDIM;k++){
	    a0[i][k]=a[i][k];
	}
    }
}

void correct_position(n,p,a,aold,dt)
    int n;
    REAL p[][NDIM];
    REAL a[][NDIM];
    REAL aold[][NDIM];
    REAL dt;
{
    register int i,k;
    register REAL dt3;
    dt3 = dt*dt/6.0;
    for(i=0;i<n;i++){
	for(k=0;k<NDIM;k++){
	    p[i][k] += dt3*(a[i][k]-aold[i][k]);
	}
    }
}

#endif

void guestimate_acc_etc( int n,
			 double eps2,
			 double m[],
			 double a[][NDIM],
			 double j[][NDIM],
			 double p[])
     /* this routine is for GRAPE-6. Gives some order-estimated values
	force etc */
{
    static int first_call = 1;
    if (first_call){
	int i,k;
	double fscale = m[0]/eps2*sqrt(n+0.0);
	double pscale = m[0]/sqrt(eps2)*sqrt(n+0.0);
	if (fscale < 10.0) fscale = 10.0;
	if (pscale < 10.0) pscale = 10.0;
	first_call = 0;
	for(i=0;i<n;i++){
	    p[i] = pscale;
	    for(k=0;k<NDIM;k++){
		a[i][k] = fscale;
		j[i][k] = fscale;
	    }
	}
    }
}



void push_system_full(n,x,v,a,pot,m,eps2, dt)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* in/out	position array */
    REAL v[][NDIM];	/* in/out	velocity array */
    REAL a[][NDIM];	/* in/out	acceleration array */
    REAL pot[];		/* in/out	potential array */
    REAL m[];		/* input	mass array */
    REAL eps2;		/* input	softening paramater squared */
    REAL dt;		/* input	timestep */
{
    REAL outer_potential();
    int i;
/*    puts("x"); print_array(n,x);*/
/*    puts("v"); print_array(n,v);*/
/*    puts("a"); print_array(n,a);*/
#ifdef PCMODE
    save_old_acc(aold,a,n);
#endif    
    push_velocity_half(n,v,a,dt);
    push_position_full(n,x,v,dt);
#ifdef EXACT_CALCULATION
    accel_accurate(n,x,m,eps2,a);
    add_outer_force(n,x,a);
#else
#ifdef CHECK_GRAPE    
    accel_accurate(n,x,m,eps2,a);
    add_outer_force(n,x,a);
    print_accel(n,a);
#endif    
#if (GRAPE1A)
    calculate_accel_by_grape1a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if (GRAPE2A)
#if 0
    calculate_accel_by_grape2a(n,x,m,a, pot,eps2);
#else
    calculate_accel_by_grape2a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#endif    
#if (HARP3)
#if !(NOSHARE)    
    h3open_();
    calculate_accel_by_harp3_separate_noopen(n,x,n,x,m,a, pot,eps2);
    h3close_();
#else
    calculate_accel_by_harp3_separate_noopen(n,x,n,x,m,a, pot,eps2);
#endif    
#endif    
#if (GRAPE6)
    {
	int clusterid = G6_CLUSTERID;
	static int g6_opened = 0;
#if !(NOSHARE)    
	g6_open_(&clusterid);
	guestimate_acc_etc(n,eps2,m,a,jerk,pot);
	calculate_accel_by_grape6_noopen(clusterid,n,x,v,m,a,jerk, pot,eps2);
	g6_close_(&clusterid);
#else
	if (!g6_opened){
	    g6_opened = 1;
	    g6_open_(&clusterid);
	}
	guestimate_acc_etc(n,eps2,m,a,jerk,pot);
	calculate_accel_by_grape6_noopen(clusterid,n,x,v,m,a,jerk, pot,eps2);
#endif
    }
#endif    
#if (GRAPE3)
    calculate_accel_by_grape3_separate(n,x,n,x,m,a, pot,eps2);
#endif
#if (GRAPE_1)    
    calculate_accel_by_grape(n,x,m,a, sqrt(eps2));
#endif
    add_outer_force(n,x,a);
#if 0
    print_accel(n,a,pot);
#endif    
#endif
    push_velocity_half(n,v,a,dt);
#ifdef PCMODE
    correct_position(n,x,a,aold,dt);
#endif
#ifdef OUTER_POTENTIAL
    for(i=0;i<n;i++){
	potchange +=
	   m[i]*(outer_potential(x[i],tsys) - outer_potential(x[i],tsys-dt));
    }
#endif    
}

#ifdef SHORTINT
int maxint=32767;
#else
#ifdef ibm
int maxint=32767;
#else
#ifdef DEC
int maxint=RAND_MAX;
#else
#ifdef IRIX
int maxint=32767;
#else
int maxint= (1<<30 - 1)*2+1;
#endif    
#endif    
#endif
#endif


REAL inverse_max_random_half;

REAL real_random();
REAL real_random()
{
    REAL dum;
    dum = rand()*inverse_max_random_half - 1.0;
#if 0
    printf("(real_random) dum = %f %x\n", dum,rand());
#endif
    return (dum);
}

void initialize_random(seed)
    int seed; 		/* input	random seed */
{
    srand(seed);
    inverse_max_random_half = 2.0/maxint;
/*    printf("(initialize_random) : maxint = %d\n", maxint);*/
}


void obtain_weighted_average(n,x,m, xave)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* input	position array */
    REAL m[];		/* input	mass array */
    double xave[NDIM];	/* output	weighted average */
{
    double msum;
    int i, k;
/*    puts("(obtain_weighted_average) entered");*/
    for(k=0; k<NDIM; k++)xave[k] = 0.0;
    msum = 0.0;
    for(i=0; i<n; i++){
    	msum += m[i];
    	for(k=0; k<NDIM; k++){
    	    xave[k] += x[i][k]*m[i];
    	}
/*    	printf("i, x, y, z : %d %f %f %f\n", i, x[i][0], x[i][1], x[i][2]);*/
    }    
/*    printf("(obtain_weighted_average) msum = %f\n", msum);*/
    for(k=0; k<NDIM; k++)xave[k] = xave[k]/msum;
/*    printf("avg.  x, y, z : %f %f %f\n",  xave[0], xave[1], xave[2]);*/
}


void adjust_average_to_zero(n,x,m)
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* in/out	position array */
    REAL m[];		/* input	mass array */
{
    double xave[NDIM];
    int i, k;
/*    puts("(adjust_average_to_zero) entered");*/
    obtain_weighted_average(n,x,m,xave);
    for(i=0; i<n; i++){
    	for(k=0; k<NDIM; k++){
    	    x[i][k] -= xave[k];
    	}
    }    
}

void set_random_vector(x)
    REAL x[NDIM];	/* output	random vector in an unit sphere */
{
    REAL sum;
    int k;
    sum = 2;
    while(sum > 1.0){
	sum = 0.0;
	for (k=0; k<NDIM; k++){
	    x[k] = real_random();
/*	    if(k == 1) x[k] = .00;*/
	    sum += x[k]*x[k];
	}
    }  
}
void create_uniform_sphere(infile,n,x,v,m)
    FILE * infile;
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* output	position array */
    REAL v[][NDIM];	/* output	velocity array */
    REAL m[];		/* output	mass array */
{
    REAL mass;
    int i, k;
    float xscale, vscale, r2, omega, vexp;
    float power;
    double pow();
    float pr, fact;
#   define FF1 "%f"
#   define FF2 "%f%f"
    fprintf(stderr, "enter scales for x and v:");
    fscanf(infile,FF2,&xscale, &vscale);
    fprintf(stderr, "enter power index for rho = r^{-p}:");
    fscanf(infile,FF1,&power);
    fprintf(stderr, "enter omega:");
    fscanf(infile,FF1,&omega);
#ifdef EXPANSION    
    fprintf(stderr, "enter vexpand:");
    fscanf(infile,FF1,&vexp);
#endif    
    printf("x, v scale factor = %f %f power = %f omega=%f\n",
	   xscale, vscale,power,omega);
#ifdef EXPANSION
    printf("expansion velocity = %f\n", vexp);
#endif    
    pr = (power+0.0)/(6-2.0*power);
/*    puts("(create_uniform_sphere) entered");*/
    initialize_random(1234567);
    mass = 1.0/n;
    for(i=0; i<n; i++){
    	m[i] = mass;
    	set_random_vector(x[i]);
    	set_random_vector(v[i]);
	r2 = x[i][0]*x[i][0] + x[i][1]*x[i][1] + x[i][2]*x[i][2];
	fact = pow(r2,pr);
    	for(k=0;k<NDIM; k++){
    		x[i][k]*=xscale*fact;
    		v[i][k]*=vscale;
    	}
#if 1
	if(omega != 0.0){
	    v[i][0] += omega*x[i][1];
	    v[i][1] -= omega*x[i][0];
	}
#endif
#ifdef EXPANSION
    	for(k=0;k<NDIM; k++){
    		v[i][k] +=x[i][k]*vexp;
    	}
#endif	
#if 0
	printf("mass[%d]=%f\n", i,m[i]);
#endif	
    }
    adjust_average_to_zero(n,x,m);
    adjust_average_to_zero(n,v,m);

}

void diag(time,n,x,v,m,pot,eps2,mode, pesample)
    REAL time;		/* input	system time */
    int  n;		/* input	number of particles */
    REAL x[][NDIM];	/* input	position array */
    REAL v[][NDIM];	/* input	velocity array */
    REAL m[];		/* input	mass array */
    REAL pot[];		/* input	potential array (GRAPE1A only) */
    REAL eps2;		/* input	softening parameter squared */
    int  mode;		/* input	energy output control
    					0 : initial (store energy)
    					1 : print DE/E */
    int pesample;	/* input	sampling flequency for 
    					potential energy calculation
    					0,1 : exact calculation */
{
    int i,j,k, k1, k2;
    double cm[NDIM], cmv[NDIM], am[NDIM];
    REAL pe, ke, etot, de;
    REAL pei, rij2, dx;
    static REAL etot0;
    REAL outer_potential();
/*    puts("(diag) entered");*/
    printf("Enter diag:"); print_cpusec();
    print_cpusec_stderr();
    /* calculate total potential energy first */
    pe = 0.0;
    if (pesample == 0) pesample = 1;
#if (GRAPE1A | GRAPE3|GRAPE2A | HARP3 |GRAPE6)
    pesample = 1;
#endif    
    for(i=0; i<n; i+=pesample){
    	pei = 0.0;
#if !((GRAPE1A)	| (GRAPE3)|GRAPE2A|HARP3 | GRAPE6)
    	for(j=i+pesample; j<n; j+=pesample){
    	    rij2 = eps2;
    	    for(k=0; k<NDIM; k++){
    	    	dx = x[i][k]-x[j][k];
    	    	rij2 += dx*dx;
/*    	    	printf("(diag) pe calc, i, j, k, xik, xjk, rij2\
: %d %d %d %f %f %f\n",
    	    	       i, j, k, x[i][k],x[j][k], rij2);*/
    	    }
    	    pei -= m[j]/sqrt(rij2);
    	}
#else
	pei = pot[i]*0.5;
#if 0
        printf("i, psum = %d, %g\n", i, pei);
#endif
#endif	
    	pe += pei*m[i];
    }
    pe = pe*pesample*pesample;
    /* calculate totoal kinetic energy */
    /* also add outer potential if specified */
    ke = 0.0;
    for(i=0; i<n; i++){
    	ke+= m[i]*(v[i][0]*v[i][0] + v[i][1]*v[i][1] + v[i][2]*v[i][2]);
#ifdef OUTER_POTENTIAL
	pe += m[i]*outer_potential(x[i],tsys);
#endif	
    }
    ke *= 0.5;
    etot = pe + ke;
    if(mode == 0){
    	etot0 = etot;
    	de = 0.0;
    }else{
    	de = (etot - etot0 - potchange)/etot;
    }
    printf("T=%12.3f  E = %15.9f  \nDE=%15.8e  V.R. = %10.6f\n", 
            time, etot, de, -ke/pe);
    fprintf(stderr,"T=%12.3f  E = %15.9f  \nDE=%15.8e  V.R. = %10.6f\n", 
            time, etot, de, -ke/pe);
    /* print center of mass etc. */
    obtain_weighted_average(n, x, m, cm);
    obtain_weighted_average(n, v, m, cmv);
    for(k=0; k<NDIM; k++){
    	am[k] = 0.0;
    	k1 = (k+1) % NDIM;
    	k2 = (k+2) % NDIM;
    	for(i=0; i<n; i++){
    	   am[k] += m[i]*(x[i][k1]*v[i][k2] - x[i][k2]*v[i][k1]);
        }
    }
    printf("CM :%12.5e %12.5e %12.5e\n",cm[0],cm[1],cm[2]);
    printf("CMV:%12.5e %12.5e %12.5e\n",cmv[0],cmv[1],cmv[2]);
    printf("AM :%12.5e %12.5e %12.5e\n",am[0],am[1],am[2]);
    printf("Exit  diag:"); print_cpusec();    
    fflush(stdout);
}

void input_parameters(infile,n, dt, tstop, dtout, dtsnapout,eps2, pesample)
    FILE * infile;	/* input        parameter file pointer */
    int * n ;		/* output	number of particles */
    REAL * dt;		/* output	timestep */
    REAL * tstop;	/* output	time to stop integration */
    REAL * dtout;	/* output	output interval */
    REAL * dtsnapout;	/* output	snapshot output interval */
    REAL * eps2;	/* output	softening parmeter squared */
    int  * pesample;	/* output	sampling frequency for potential
    					calculation */
{

    REAL eps;
    fprintf(stderr,"enter n, dt, tstop, dtout, dtsnapout, eps, pesample:\n");
#ifdef SHORTREAL    
    printf("Reading FLOAT parameters....\n");
    fscanf(infile,"%d%f%f%f%f%f%d", n, dt, tstop,
	   dtout, dtsnapout, &eps,pesample);
#else
    fprintf(stderr,"LONG  format read\n");
#ifndef LFFORMAT
    fscanf(infile,"%d%F%F%F%F%F%d", n, dt, tstop, dtout,
	   dtsnapout, &eps,pesample);
#else
    printf("Reading DOUBLE parameters....\n");
    fscanf(infile,"%d%lf%lf%lf%lf%lf%d", n, dt, tstop, dtout, dtsnapout, &eps,pesample);
#endif    
#endif
    *eps2 = eps*eps;
    printf("n=%4d dt=%7.4f tstop=%8.3f dtout=%7.3f dtsnapout=%7.3f\n\
eps=%8.5f pesample=%d\n",
           *n, *dt, *tstop, *dtout, *dtsnapout, eps,*pesample);
}
#ifdef YAPP_SV
scrdump(frame,frameno)
    string frame;    
    int frameno;
{
    char fname[64];
    if((frame != NULL) && (frame[0] != 0)){
	sprintf(fname, "%s.%d", frame, frameno);
	pl_screendump(fname); /*patch by Jun, 90/1/11*/
    }
}
#endif


void scale_uniform_sphere(n,eps2,x,v,m,a,pot)
    int n;
    REAL eps2;
    REAL x[][NDIM];
    REAL v[][NDIM];
    REAL m[];
    REAL a[][NDIM];
    REAL pot[];
{
    int i,k;
    double etot;
    double sqrt();
    double rscale, vscale;
#if 0
    for(i=0;i<n;i++){
	printf("x[%d]=%f %f %f\n", i,x[i][0],x[i][1],x[i][2]);
	printf("mass[%d]=%f\n", i,m[i]);
    }
#endif
#if (GRAPE1A)
    calculate_accel_by_grape1a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if (GRAPE2A)
    calculate_accel_by_grape2a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if (HARP3)
    h3open_();
    calculate_accel_by_harp3_separate_noopen(n,x,n,x,m,a, pot,eps2);
    h3close_();
#endif    
#if (GRAPE6)
    {
	int clusterid = G6_CLUSTERID;
	guestimate_acc_etc(n,eps2,m,a,jerk,pot);
	g6_open_(&clusterid);
	
	calculate_accel_by_grape6_noopen(clusterid,n,x,v,m,a,jerk, pot,eps2);
	g6_close_(&clusterid);
    }
#endif    
#if (GRAPE3)
    calculate_accel_by_grape3_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if  (GRAPE_1)
    return;
#endif
    etot = 0.0;
    for(i=0;i<n;i++){
#if 0
	printf("mass[%d]=%f, a, pot=%e %e %e %e\n", i,m[i],
	       a[i][0], a[i][1], a[i][2],pot[i]);
#endif	
	etot += m[i]*(v[i][0]*v[i][0]+v[i][1]*v[i][1]+v[i][2]*v[i][2]
		      +pot[i]);
#if 0	
	printf("pot[%d]=%f %f\n", i, pot[i], a[i][0]);
#endif	
    }
    etot *= 0.5;
    rscale = -etot/0.25;
    vscale = sqrt(1.0/rscale);
    printf("Etot = %f %f %f\n", etot,rscale,vscale);
    for(i=0;i<n;i++){
	for(k=0;k<NDIM;k++){
	    x[i][k] *= rscale;
	    v[i][k] *= vscale;
	}
    }
}
static REAL x[NMAX][NDIM];
static REAL v[NMAX][NDIM];
static REAL m[NMAX];
static REAL a[NMAX][NDIM];
static REAL pot[NMAX];
static FILE * input_parameter_file;
#ifdef NEMO
string frame;
#endif
main(argc,argv)
    int argc;
    char *argv[];
{
    static int  n;
    int pot_out_flag = 0;
    int frameno = 0;
    static REAL dt, tstop, dtout, dtsnapout, eps2;
    static REAL time, tnext, tsnapnext;
    int pesample;

    timer_init();
#ifdef NEMOOUT
    initparam(argv,defv);
#endif    
#ifndef EXACT_CALCULATION
#if NOSHARE
#   if !(GRAPE1A | GRAPE3 | GRAPE2A|HARP3|GRAPE6)
    printf("initialize_grape, retcode = %d\n", initialize_grape());
#   endif
#   if (GRAPE1A)
    printf("initialize_grape1a, retcode = %d\n", initialize_grape1a());
#   endif    
#   if GRAPE3
    printf("initialize_grape3, retcode = %d\n", initialize_grape3());
#   endif
#   if GRAPE2A
    printf("initialize_grape2a, retcode = %d\n", initialize_grape2a());
        grape2a_reset();
    setup_grape2a_for_gravity();

#   endif
#   if GRAPE6
    {
	int clusterid = G6_CLUSTERID;
	printf("initialize_grape6, retcode = %d\n", g6_open_(&clusterid));
    }
#   endif
#endif
#endif
#if (GRAPE1A | GRAPE3 | GRAPE2A|HARP3|GRAPE6)    
    pot_out_flag = 1;
#endif    
#ifndef NEMOOUT    
    printf("argc = %d, argv[1]=%s\n", argc, argv[1]);
    if(argc <= 1){
	fprintf(stderr,"No parameter specified: read from stdin\n");
	input_parameter_file=stdin;
    }else{
	input_parameter_file = fopen(argv[1], "r");
    }
#else
    input_parameter_file = stropen(getparam("in"),"r");
#endif    
#ifdef YAPP_SV
    frame = getparam("frame");
#endif    
    input_parameters(input_parameter_file,
		     &n, &dt, &tstop, &dtout,
		     &dtsnapout, &eps2,&pesample);
    if(set_output_snap_file(input_parameter_file)){
    	puts("snap file open error");
    	exit(1);
    }
    if(n==0){
    	if(set_input_file(input_parameter_file)){
    		puts("snap file open error");
    		exit(1);
    	}
    	if(read_snap(&n,&time,m,x,v,NMAX)){
    		puts("snap file read error");
    		exit(1);
    	}
#if GRAPE3
#if !(NOSHARE)
    initialize_grape3();
    free_grape3();
#endif    
    set_scales(1.0/512,m[0]);
#endif    
    }else{
	time = 0.0;
    	create_uniform_sphere(input_parameter_file,n,x,v,m);
#if GRAPE3
#if !(NOSHARE)
    initialize_grape3();
    free_grape3();
#endif    
    set_scales(1.0/512,m[0]);
#endif    
	scale_uniform_sphere(n,eps2,x,v,m,a,pot);
    }
    tsys = time;
#ifdef OUTER_POTENTIAL
    set_plummer_field(input_parameter_file);
    set_scale_time(input_parameter_file);
#endif
    initialize_plot(input_parameter_file);    
    tnext = time + dtout - 0.5*dt;
    tsnapnext = time + dtsnapout - 0.5 * dt;
    plot_particles(n,x,time);
#if YAPP_SV
    scrdump(frame,frameno);
    frameno++;
#endif    
#ifdef EXACT_CALCULATION
    accel_accurate(n,x,m,eps2,a);
    add_outer_force(n,x,a);
#else    
#ifdef CHECK_GRAPE        
    accel_accurate(n,x,m,eps2,a);
    add_outer_force(n,x,a);
    print_accel(n,a);
#endif    
#if (GRAPE1A)
    calculate_accel_by_grape1a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if (GRAPE2A)
    calculate_accel_by_grape2a_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if (HARP3)
    h3open_();
    calculate_accel_by_harp3_separate_noopen(n,x,n,x,m,a, pot,eps2);
#ifndef NOSHARE    
    h3close_();
#endif    
#endif    
#if (GRAPE6)
    {
	int clusterid = G6_CLUSTERID;
	g6_open_(&clusterid);
	guestimate_acc_etc(n,eps2,m,a,jerk,pot);
	calculate_accel_by_grape6_noopen(clusterid,n,x,v,m,a,jerk, pot,eps2);
	g6_close_(&clusterid);
    }
#endif    
#if (GRAPE3)
    calculate_accel_by_grape3_separate(n,x,n,x,m,a, pot,eps2);
#endif    
#if  (GRAPE_1)
    calculate_accel_by_grape(n,x,m,a, sqrt(eps2));
#endif
    add_outer_force(n,x,a);
    diag(time,n,x,v,m,pot,eps2, 0, pesample);
    out_snap_binary(n,time,m,x,v,pot,pot_out_flag);
#if 0
    print_accel(n,a,pot);
#endif    
#endif
    while(time < tstop){
#ifdef USE_STOPFILE
        int waited;
        waited = 0;
        while(check_stopfile() == 0){
           if(!waited)fprintf(stderr,"Blocked - remove /tmp/stop_grape3a to resume");
           waited = 1;
           fprintf(stderr,".");
           sleep(1);
	}
        if(waited)fprintf(stderr,"\n");
#endif
        push_system_full(n,x,v,a,pot,m,eps2, dt);
        time += dt;
	tsys = time;
        if(time >= tsnapnext){
#ifndef CONTORPLOT	    
	    plot_particles(n,x,time);
#else
	    contor_particles(n,x,time);
#endif	    
	    
            tsnapnext += dtsnapout;
	    out_snap_binary(n,time,m,x,v,pot,pot_out_flag);
#if YAPP_SV
	    scrdump(frame,frameno);
	    frameno++;
#endif    
        }
        if(time >= tnext){
            tnext += dtout;
            diag(time,n,x,v,m,pot,eps2, 1,pesample);
        }
    }
#if NOSHARE
#   if (GRAPE1A)
    free_grape1a();
#   endif
#if GRAPE3    
    free_grape3();
#endif
#if GRAPE2A
    free_grape2a();
#   endif
#if HARP3
    h3close_();
#   endif
#if (GRAPE6)
    g6close(G6_CLUSTERID);
#endif    
#endif
}
    
